<?php
/**
 * کلاس مدیریت Session
 * Session Management Class
 */

class Session {
    private static $instance = null;
    private $sessionStarted = false;

    /**
     * Constructor
     */
    private function __construct() {
        $this->start();
    }

    /**
     * Singleton Pattern
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * شروع Session
     */
    public function start() {
        if (!$this->sessionStarted && session_status() === PHP_SESSION_NONE) {
            // تنظیمات امنیتی Session
            ini_set('session.cookie_httponly', 1);
            ini_set('session.use_only_cookies', 1);
            ini_set('session.cookie_samesite', 'Strict');

            if (!DEBUG_MODE) {
                ini_set('session.cookie_secure', 1); // فقط HTTPS
            }

            session_start();
            $this->sessionStarted = true;

            // بررسی امنیت Session
            $this->validateSession();
        }
    }

    /**
     * اعتبارسنجی Session
     */
    private function validateSession() {
        // بررسی IP Address
        if ($this->has('ip_address')) {
            $currentIp = $_SERVER['REMOTE_ADDR'] ?? '';
            if ($this->get('ip_address') !== $currentIp) {
                $this->destroy();
                return;
            }
        }

        // بررسی User Agent
        if ($this->has('user_agent')) {
            $currentAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
            if ($this->get('user_agent') !== $currentAgent) {
                $this->destroy();
                return;
            }
        }

        // تنظیم مقادیر اولیه
        if (!$this->has('ip_address')) {
            $this->set('ip_address', $_SERVER['REMOTE_ADDR'] ?? '');
        }

        if (!$this->has('user_agent')) {
            $this->set('user_agent', $_SERVER['HTTP_USER_AGENT'] ?? '');
        }
    }

    /**
     * تنظیم مقدار
     */
    public function set($key, $value) {
        $_SESSION[$key] = $value;
    }

    /**
     * دریافت مقدار
     */
    public function get($key, $default = null) {
        return $_SESSION[$key] ?? $default;
    }

    /**
     * بررسی وجود کلید
     */
    public function has($key) {
        return isset($_SESSION[$key]);
    }

    /**
     * حذف مقدار
     */
    public function remove($key) {
        if (isset($_SESSION[$key])) {
            unset($_SESSION[$key]);
        }
    }

    /**
     * دریافت همه مقادیر
     */
    public function all() {
        return $_SESSION;
    }

    /**
     * پاک کردن همه مقادیر
     */
    public function clear() {
        $_SESSION = [];
    }

    /**
     * تولید مجدد Session ID
     */
    public function regenerate($deleteOld = true) {
        if ($this->sessionStarted) {
            session_regenerate_id($deleteOld);
        }
    }

    /**
     * نابود کردن Session
     */
    public function destroy() {
        if ($this->sessionStarted) {
            $_SESSION = [];

            // حذف Session Cookie
            if (isset($_COOKIE[session_name()])) {
                setcookie(
                    session_name(),
                    '',
                    time() - 3600,
                    '/',
                    '',
                    !DEBUG_MODE,
                    true
                );
            }

            session_destroy();
            $this->sessionStarted = false;
        }
    }

    /**
     * تنظیم Flash Message
     */
    public function setFlash($key, $message, $type = 'info') {
        $_SESSION['flash'][$key] = [
            'message' => $message,
            'type' => $type
        ];
    }

    /**
     * دریافت Flash Message
     */
    public function getFlash($key) {
        if (isset($_SESSION['flash'][$key])) {
            $flash = $_SESSION['flash'][$key];
            unset($_SESSION['flash'][$key]);
            return $flash;
        }
        return null;
    }

    /**
     * بررسی وجود Flash Message
     */
    public function hasFlash($key) {
        return isset($_SESSION['flash'][$key]);
    }

    /**
     * دریافت همه Flash Messages
     */
    public function getAllFlash() {
        $flashes = $_SESSION['flash'] ?? [];
        unset($_SESSION['flash']);
        return $flashes;
    }

    /**
     * تنظیم پیام موفقیت
     */
    public function setSuccess($message) {
        $this->setFlash('success', $message, 'success');
    }

    /**
     * تنظیم پیام خطا
     */
    public function setError($message) {
        $this->setFlash('error', $message, 'danger');
    }

    /**
     * تنظیم پیام هشدار
     */
    public function setWarning($message) {
        $this->setFlash('warning', $message, 'warning');
    }

    /**
     * تنظیم پیام اطلاعات
     */
    public function setInfo($message) {
        $this->setFlash('info', $message, 'info');
    }

    /**
     * جلوگیری از Clone
     */
    private function __clone() {}

    /**
     * جلوگیری از Unserialize
     */
    public function __wakeup() {
        throw new Exception("Cannot unserialize singleton");
    }
}

?>