<?php
/**
 * Helper Functions
 * توابع کمکی عمومی
 */

/**
 * Escape HTML
 */
function h($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

/**
 * Redirect
 */
function redirect($url) {
    header("Location: {$url}");
    exit;
}

/**
 * تبدیل تاریخ میلادی به شمسی
 */
function toJalali($date) {
    if (!$date) return '';

    $timestamp = strtotime($date);
    if (!$timestamp) return $date;

    // اینجا باید کتابخانه تبدیل تاریخ استفاده شود
    // برای سادگی فعلا همان میلادی را برمی‌گردانیم
    return date('Y/m/d H:i', $timestamp);
}

/**
 * فرمت تاریخ فارسی
 */
function formatDate($date, $format = 'Y/m/d H:i:s') {
    if (!$date) return '-';
    $timestamp = strtotime($date);
    if (!$timestamp) return $date;
    return date($format, $timestamp);
}

/**
 * تبدیل عدد به فارسی
 */
function toPersianNumber($number) {
    $persian = ['۰', '۱', '۲', '۳', '۴', '۵', '۶', '۷', '۸', '۹'];
    $english = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
    return str_replace($english, $persian, $number);
}

/**
 * فرمت حجم فایل
 */
function formatBytes($bytes, $precision = 2) {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];

    for ($i = 0; $bytes > 1024; $i++) {
        $bytes /= 1024;
    }

    return round($bytes, $precision) . ' ' . $units[$i];
}

/**
 * تولید رشته تصادفی
 */
function generateRandomString($length = 16) {
    return bin2hex(random_bytes($length / 2));
}

/**
 * اعتبارسنجی ایمیل
 */
function isValidEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

/**
 * اعتبارسنجی رمز عبور
 */
function isValidPassword($password) {
    return strlen($password) >= PASSWORD_MIN_LENGTH;
}

/**
 * Hash کردن رمز عبور
 */
function hashPassword($password) {
    return password_hash($password, PASSWORD_BCRYPT, ['cost' => 12]);
}

/**
 * بررسی رمز عبور
 */
function verifyPassword($password, $hash) {
    return password_verify($password, $hash);
}

/**
 * تولید CSRF Token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * بررسی CSRF Token
 */
function verifyCSRFToken($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * نمایش پیام Flash
 */
function setFlashMessage($type, $message) {
    $_SESSION['flash_message'] = [
        'type' => $type, // success, error, warning, info
        'message' => $message
    ];
}

/**
 * دریافت و حذف پیام Flash
 */
function getFlashMessage() {
    if (isset($_SESSION['flash_message'])) {
        $message = $_SESSION['flash_message'];
        unset($_SESSION['flash_message']);
        return $message;
    }
    return null;
}

/**
 * Sanitize Input
 */
function sanitize($input) {
    if (is_array($input)) {
        return array_map('sanitize', $input);
    }
    return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
}

/**
 * JSON Response
 */
function jsonResponse($data, $statusCode = 200) {
    http_response_code($statusCode);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Success Response
 */
function successResponse($data = [], $message = 'موفق') {
    jsonResponse([
        'success' => true,
        'message' => $message,
        'data' => $data
    ]);
}

/**
 * Error Response
 */
function errorResponse($message = 'خطا', $code = 400, $details = []) {
    jsonResponse([
        'success' => false,
        'error' => [
            'message' => $message,
            'details' => $details
        ]
    ], $code);
}

/**
 * Pagination
 */
function paginate($total, $page = 1, $perPage = ITEMS_PER_PAGE) {
    $totalPages = ceil($total / $perPage);
    $page = max(1, min($page, $totalPages));
    $offset = ($page - 1) * $perPage;

    return [
        'total' => $total,
        'per_page' => $perPage,
        'current_page' => $page,
        'total_pages' => $totalPages,
        'offset' => $offset,
        'has_prev' => $page > 1,
        'has_next' => $page < $totalPages
    ];
}

/**
 * تولید لینک Pagination
 */
function paginationLinks($pagination, $baseUrl) {
    $html = '<ul class="pagination">';

    // Previous
    if ($pagination['has_prev']) {
        $prev = $pagination['current_page'] - 1;
        $html .= '<li><a href="' . $baseUrl . '?page=' . $prev . '">قبلی</a></li>';
    }

    // صفحات
    for ($i = 1; $i <= $pagination['total_pages']; $i++) {
        $active = ($i === $pagination['current_page']) ? 'class="active"' : '';
        $html .= '<li ' . $active . '><a href="' . $baseUrl . '?page=' . $i . '">' . $i . '</a></li>';
    }

    // Next
    if ($pagination['has_next']) {
        $next = $pagination['current_page'] + 1;
        $html .= '<li><a href="' . $baseUrl . '?page=' . $next . '">بعدی</a></li>';
    }

    $html .= '</ul>';
    return $html;
}

/**
 * Get User Status Badge
 */
function getUserStatusBadge($status) {
    $badges = [
        'active' => '<span class="badge bg-green">فعال</span>',
        'suspended' => '<span class="badge bg-orange">تعلیق</span>',
        'banned' => '<span class="badge bg-red">مسدود</span>',
        'deleted' => '<span class="badge bg-grey">حذف شده</span>'
    ];
    return $badges[$status] ?? '<span class="badge bg-grey">' . $status . '</span>';
}

/**
 * Get Plan Badge
 */
function getPlanBadge($plan) {
    $badges = [
        'free_trial' => '<span class="badge bg-grey">آزمایشی</span>',
        'freemium' => '<span class="badge bg-blue">رایگان</span>',
        'plus' => '<span class="badge bg-orange">پلاس</span>',
        'premium' => '<span class="badge bg-purple">پرمیوم</span>'
    ];
    return $badges[$plan] ?? '<span class="badge">' . $plan . '</span>';
}

/**
 * Time Ago
 */
function timeAgo($datetime) {
    $timestamp = strtotime($datetime);
    if (!$timestamp) return $datetime;

    $diff = time() - $timestamp;

    if ($diff < 60) {
        return 'لحظاتی پیش';
    } elseif ($diff < 3600) {
        $mins = floor($diff / 60);
        return $mins . ' دقیقه پیش';
    } elseif ($diff < 86400) {
        $hours = floor($diff / 3600);
        return $hours . ' ساعت پیش';
    } elseif ($diff < 604800) {
        $days = floor($diff / 86400);
        return $days . ' روز پیش';
    } else {
        return formatDate($datetime, 'Y/m/d');
    }
}

/**
 * فرمت پول
 */
function formatMoney($amount, $currency = 'EUR') {
    $symbols = [
        'EUR' => '€',
        'USD' => '$',
        'IRR' => 'تومان'
    ];

    $symbol = $symbols[$currency] ?? $currency;
    return number_format($amount, 2) . ' ' . $symbol;
}

/**
 * Get Gender Label
 */
function getGenderLabel($gender) {
    $labels = [
        'male' => 'مرد',
        'female' => 'زن',
        'other' => 'سایر',
        'prefer_not_to_say' => 'ترجیح می‌دهم نگویم'
    ];
    return $labels[$gender] ?? '-';
}

/**
 * Debug Output
 */
function debug($var, $die = false) {
    if (DEBUG_MODE) {
        echo '<pre style="direction:ltr; text-align:left; background:#f5f5f5; padding:10px; border:1px solid #ddd;">';
        print_r($var);
        echo '</pre>';
        if ($die) die();
    }
}

/**
 * Log Error
 */
function logError($message, $file = 'error.log') {
    $logPath = ROOT_PATH . 'logs/' . $file;
    $timestamp = date('Y-m-d H:i:s');
    $logMessage = "[{$timestamp}] {$message}" . PHP_EOL;

    if (!is_dir(ROOT_PATH . 'logs')) {
        mkdir(ROOT_PATH . 'logs', 0755, true);
    }

    file_put_contents($logPath, $logMessage, FILE_APPEND);
}

/**
 * بررسی AJAX Request
 */
function isAjax() {
    return !empty($_SERVER['HTTP_X_REQUESTED_WITH']) &&
           strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest';
}

/**
 * Get Request Method
 */
function getRequestMethod() {
    return $_SERVER['REQUEST_METHOD'] ?? 'GET';
}

/**
 * Is POST Request
 */
function isPost() {
    return getRequestMethod() === 'POST';
}


/**
 * Get POST Data
 */
function post($key, $default = null) {
    return $_POST[$key] ?? $default;
}

/**
 * Get GET Data
 */
function get($key, $default = null) {
    return $_GET[$key] ?? $default;
}

/**
 * Get Role Label
 */
function get_role_label($role) {
    $roles = [
        'super_admin' => '<span class="label bg-purple">مدیر ارشد</span>',
        'admin' => '<span class="label bg-blue">مدیر</span>',
        'moderator' => '<span class="label bg-cyan">ناظر</span>'
    ];
    return $roles[$role] ?? $role;
}

/**
 * Truncate Text
 */
function truncate($text, $length = 50, $suffix = '...') {
    if (mb_strlen($text) <= $length) {
        return $text;
    }
    return mb_substr($text, 0, $length) . $suffix;
}
